function slml2report( varargin )
% SLML2REPORT Generate a report comparing various SLML models.
%
% COMMENT: When the method is not deployed the shape of the input arguments is
% slml2report( models, param )
%
% List Of Input Arguments  Descriptions
% -----------------------  ------------
% models                   A cell array of filenames
% param                    A structure holding the function options
%
% The shape of param is described
%
% List Of Parameters        Descriptions
% ------------------        ------------
% verbose                   (optional) Print the intermediate steps to screen. Default is true.
% resolution.cell           (optional) The resolution of the cell and nucleus that are being passed in
% resolution.objects        (optional) The resolution of the object model being synthesized
% includenuclear            (optional) Include parameters of the nuclear model in the comparison
% includecell               (optional) Include parameters of the cell model in the comparison
% includeprot               (optional) Include parameters of the protein model in the comparison
%

% Author: Robert F. Murphy
% Created: June 29, 2013

% Copyright (C) 2013 Murphy Lab
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

% if ~isdeployed
%     if nargin ~= 2
%         error( ['CellOrganizer: Wrong number of input arguments. If slml2img ' ...
%             ' is not deployed, then the number of input arguments must be 2.'] );
%     end
%
% end

%INPUT ARGUMENTS
if nargin < 2
    param = [];
else
    param = varargin{2};
end    
param = ml_initparam(param,struct('verbose',true,'includenuclear',true,'includecell',true,'includeprot',true));

if ~(param.includenuclear | param.includecell | param.includeprot) error('At least one set of parameters must be included.'); end

%check if files exists and are valid SLML instances
%parse SLML instances into Matlab structures

filenames = varargin{1};

%if exist('testing.mat', 'file')
%    load testing
%else
    
if param.verbose
    fprintf( 1, '\n%s%%', 'Parsing SLML instances: 0' );
end

try
    for j=1:1:length(filenames)
        %if it is a mat file then load directly into memory
        if( isMatFile(filenames{j}) )
            load(filenames{j});
            models{j} = model;
        elseif isstruct(filenames{j})
            models{j} = filenames{j};
        else
            models{j} = slml2model( filenames{j} );
        end
        
        
        n=length(filenames);
        if param.verbose
            %indicate an object has been successfully added
            fprintf( 1, [ repmat('\b', 1, length(num2str(100*(j-1)/n))+1) '%s'], ...
                repmat(' ', 1, length(num2str(100*(j-1)/n))+1) );
            fprintf( 1, [ repmat('\b', 1, length(num2str(100*(j-1)/n))+1) '%s%%'], num2str(100*j/n) );
            pause(0.1)
        end
    end
    fprintf( 1, '\n', 0 );
catch err
    warning( 'CellOrganizer: Unable to parse SLML instances. Exiting program.' );
        
    if ~isdeployed
        return
    else
        exit
    end
end

try
    classlabels = varargin{3};
catch
%
% use the class labels for the various models as the independent
% (categorical) variable
%
    classlabels = [];
    for j=1:n
        try
            classlabels = strcat(classlabels, [models{j}.proteinShape.class ';']);
        catch
            classlabels = strcat(classlabels, [models{j}.proteinShape.type ';']);
        end
    end
    classlabels=classlabels(1:end-1);
end

%
% generate reports
%

try
    indvar = varargin{4};
    models2report(models,param,classlabels,indvar)
catch
    models2report(models,param,classlabels)
end
