function superid = generate_superid_from_model_file( model_file )
%GENERATE_SUPERID_FROM_MODEL_FILE Helper function that generates the valid
%model file superid given a Matlab model file.

% Copyright (C) 2007-2014 Murphy Lab
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

superid = '';

if nargin ~= 1
	error( 'Wrong number of input arguments' );
elseif ~isa( model_file, 'char' )
	error( 'Input argument must be a string' )
elseif ~exist( model_file )
	error( [ 'File ' model_file ' does not exist.' ] );
else
	load( model_file )
end

%cell line
if isfield( model, 'cellline' )
	superid = [ superid 'cellline-' lower(model.cellline) ];
else
	superid = [ superid 'cellline-unknown' ];
end

%model dimensionality
if isfield( model, 'dimensionality' )
	superid = [ superid '.dimensionality-' upper(model.dimensionality) ];
else
	superid = [ superid '.dimensionality-unknown' ];
end

if isfield( model, 'dimensionality' )
    if strcmpi( model.dimensionality, '2D' )
        %protein model class
        if isfield( model, 'proteinModel' ) && isfield( model.proteinModel, 'class' )
            superid = [ superid '.protein_model_class-' lower(model.proteinModel.class) ];
        else
            superid = [ superid '.protein_model_class-unknown' ];
        end
        
        %protein model type
        if isfield( model, 'proteinModel' ) && isfield( model.proteinModel, 'type' )
            superid = [ superid '.protein_model_class-' lower(model.proteinModel.type) ];
        else
            superid = [ superid '.protein_model_type-unknown' ];
        end
        
    else strcmpi( model.dimensionality, '3D' )
        %nuclear shape class
        if isfield( model, 'nuclearShapeModel' ) && isfield( model.nuclearShapeModel, 'class' )
            superid = [ superid '.nuclear_model_class-' ...
                regexprep(lower(model.nuclearShapeModel.class), ' ', '_') ];
        elseif ~isfield( model, 'nuclearShapeModel' )
            superid = [ superid '.nuclear_model_class-none' ];
        else
            superid = [ superid '.nuclear_model_class-unknown' ];
        end
        
        %nuclear shape type
        if isfield( model, 'nuclearShapeModel' ) && isfield( model.nuclearShapeModel, 'type' )
            superid = [ superid '.nuclear_model_type-' ...
                regexprep(lower(model.nuclearShapeModel.type), ' ', '_' ) ];
        elseif ~isfield( model, 'nuclearShapeModel' )
            superid = [ superid '.nuclear_model_type-none' ];
        else
            superid = [ superid '.nuclear_model_type-unknown' ];
        end
        
        %cell shape class
        if isfield( model, 'cellShapeModel' ) && isfield( model.cellShapeModel, 'class' )
            superid = [ superid '.cell_model_class-' ...
                regexprep(lower(model.cellShapeModel.class), ' ', '_') ];
        elseif ~isfield( model, 'celShapeModel' )
            superid = [ superid '.cell_model_class-none' ];
        else
            superid = [ superid '.cell_model_class-unknown' ];
        end
        
        %cell shape type
        if isfield( model, 'cellShapeModel' ) && isfield( model.cellShapeModel, 'type' )
            superid = [ superid '.cell_model_type-' ...
                regexprep(lower(model.cellShapeModel.type), ' ', '_' ) ];
        elseif ~isfield( model, 'cellShapeModel' )
            superid = [ superid '.cell_model_type-none' ];
        else
            superid = [ superid '.model_type-unknown' ];
        end
        
        %protein model class
        if isfield( model, 'proteinShape' ) && isfield( model.proteinShape, 'class' )
            superid = [ superid '.protein_model_class-' lower(model.proteinShape.class) ];
        elseif ~isfield( model, 'proteinShape' )
            superid = [ superid '.protein_model_class-none' ];
        else
            superid = [ superid '.protein_model_class-unknown' ];
        end
        
        %protein model type
        if isfield( model, 'proteinShape' ) && isfield( model.proteinShape, 'type' )
            superid = [ superid '.protein_model_class-' lower(model.proteinShape.type) ];
        elseif ~isfield( model, 'proteinShape' )
            superid = [ superid '.protein_model_class-none' ];
        else
            superid = [ superid '.protein_model_type-unknown' ];
        end
        
    end
else
    %protein model class
    if isfield( model, 'proteinShape' ) && isfield( model, 'proteinModel' )
        superid = [ superid '.protein_model_class-none' ];
    end
end

%author
if isfield( model, 'documentation' ) && isfield( model.documentation, 'author' )
	superid = [ superid '.author-' lower(model.documentation.author) ];
else
	superid = [ superid '.author-unknown' ];
end

%version
if isfield( model, 'version' )
	superid = [ superid '.version-' lower(model.version) ];
else
	superid = [ superid '.version-unknown' ];
end


