function demo3D25(param)
% demo3D25
%
% Trains a generative model if microtubules

% Created: Ivan E. Cao-Berg & Gregory R. Johnson
%
% Copyright (C) 2014 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu
    
curr_path = which([mfilename '.m']);
curr_path = curr_path(1:end-(length([mfilename '.m'])));
cd(curr_path);

%grj 10/31/14
frameworkflag = true;


if ~exist('param', 'var')
    param = [];
end

if ~isa( frameworkflag, 'logical' )
   warning( 'Input variable framework flag must be boolean. Exiting demo.' );
   return
end

pattern = 'Tub';
imgdir = [ pwd filesep pattern ];
param_default.protein.class = 'microtubule';
param_default.protein.name = 'Tub';
param_default.nucleus.name = 'Tub';
param_default.cell.model = 'Tub';


tarball = [ '3DHela_' pattern '.tgz' ];
url = 'http://murphylab.web.cmu.edu/data/Hela/3D/multitiff';
   
if ~exist( [pwd filesep tarball], 'file')
    disp('Downloading dataset');
    urlwrite([url filesep tarball], tarball);
    disp('Extracting files');
    untar( tarball );
else
    disp('Pattern found on disk. Extracting files from existing tarball');
    if ~exist( imgdir )
        disp('Extracting files');
        system(['tar -xvf ' tarball] );
    else
        disp('Directory found. Skipping decompression');
    end
end

dimensionality = '3D';
answer = false;




dna = [imgdir filesep 'cell*ch0*.tif'] ;
% cell = [directory filesep 'cell*ch1*.tif'];
cell = [imgdir filesep 'cell*ch1*.tif'];
protein = [imgdir filesep 'cell*ch2*.tif'];
mask = [imgdir filesep 'cell*mask*.tif'];


% param.masks = [directory filesep 'cell*mask*.tif'];

% param.masks = [directory filesep 'cell*mask*.tif'];

% generic model options
% ---------------------
% model.name                (optional) Holds the name of the model. Default is empty.
param_default.model.name = 'all';

% model.id                  (optional) Holds the id of the model. Default is empty.
try
   [status,result] = system( 'uuidgen' ); 
   param_default.model.id = result(1:end-1);
catch
   param_default.model.id = num2str(now);
end

% model.filename            Holds the output filename.
param_default.model.filename = [ lower(pattern) '.xml' ];

% nuclear shape model options
% ---------------------------
% nucleus.type              Holds the nuclear model type. Default is "major axis".
param_default.nucleus.type = 'major axis';

% nucleus.id                (optional) Holds the id of the nuclear model. Default is empty.
try
   [status,result] = system( 'uuidgen' ); 
   param_default.nucleus.id = result(1:end-1);
catch
   param_default.nucleus.id = num2str(now);
end

% cell shape model options
% ------------------------
% cell.type                 Holds the cell model type. Default is "ratio".
param_default.cell.type = 'ratio';

% cell.id                   (optional) Holds the id the cell model. Default is empty.
try
   [status,result] = system( 'uuidgen' ); 
   param_default.cell.id = result(1:end-1);
catch
   param_default.cell.id = num2str(now);
end

% protein shape model options
% ---------------------------
% protein.type              (optional) Holds the protein model type. The default is "vesicle".
param_default.protein.type = 'network';

% protein.id                (optional) Holds the id of the protein model. The default is empty.
try
   [status,result] = system( 'uuidgen' ); 
   param_default.protein.id = result(1:end-1);
catch
   param_default.protein.id = num2str(now);
end

% protein.cytonuclearflag   (optional) Determines whether the protein pattern will be generated in
%                           the cytosolic space ('cyto'), nuclear space ('nuc') or everywhere ('all').
%                           Default is cyto.
if strcmpi( param_default.protein.class, 'nuc' )
   param_default.protein.cytonuclearflag = 'nuc';
else
   param_default.protein.cytonuclearflag = 'cyto';
end

% other options
% -------------
% verbose                   (optional) Displays messages to screen. The default is true.
param_default.verbose = true;

% debug                     (optional) Reports errors and warnings. Default is false.
param_default.debug = true;

% train.flag                (optional) Selects what model is going to be trained ('nuclear',
%                           'framework', or 'all'). Default is 'all'
param_default.train.flag = 'microtubule';

%documentation
param_default.documentation.description = 'This model has been trained using demo3D12 from CellOrganizer';

%model resolution
%this is the resolution of the datasets
param_default.model.resolution = [0.049, 0.049, 0.2000];

param_default.model.downsampling = [4,4,1];

param_default.saveIntermediateResults = false;

param = ml_initparam(param, param_default);


dna_paths = dir(dna);
dna_paths = sort_nat({dna_paths.name});

prot_paths = dir(protein);
prot_paths = sort_nat({prot_paths.name});

cell_paths = dir(cell);
cell_paths = sort_nat({cell_paths.name});

mask_paths = dir(mask);
mask_paths = sort_nat({mask_paths.name});

for i = 1:length(dna_paths)
    cell_imgs{i} = @() seg_hela([imgdir filesep cell_paths{i}], [imgdir filesep mask_paths{i}]);
    dna_imgs{i} = @() seg_hela([imgdir filesep dna_paths{i}], [imgdir filesep mask_paths{i}]);
    prot_imgs{i} = [imgdir filesep prot_paths{i}];
end

if ~exist([pwd filesep 'tub.mat'], 'file')
    answer = img2slml( dimensionality, dna_imgs, cell_imgs, prot_imgs, param );
else
    answer = true;
end

if answer
    load([pwd filesep 'tub.mat']) %load the tubulin model we just built
    model.proteinShape.resolution =  [0.049, 0.049, 0.2000].*[4,4,1];
    tub = model;
    load(['centro.mat']) %load the old centrosome model
    centro = model;
    
    img = model2img({tub, centro});
    save('tubimg.mat', 'img')
end

