function answer = demo3D11( param )
% demo3D11
%
% Trains a generative model of the cell framework using the four patterns in the 3D HeLa
% dataset from the Murphy Lab

% Created: Ivan E. Cao-Berg
%
% Copyright (C) 2012-2104 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% May 15, 2013 I. Cao-Berg Updated method to support wildcards
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

curr_path = which('demo3D11.m');
curr_path = curr_path(1:end-10);
cd(curr_path);

if nargin == 0
    param = [];
    model = [];
else
    try
        model = param.model;
    catch
        model = [];
    end
end

model = ml_initparam( model, ...
    struct( 'downsampling', [1,1,1] ) );
param = ml_initparam( param, ...
    struct( 'seed', 3 ) );
param = ml_initparam( param, ...
    struct( 'preprocess', true ) );

param.model = model;
clear model

%list of available patterns
patterns = { 'Nuc', 'Mit', 'LAM', 'TfR' };
% patterns = {'TfR'};

%icaoberg 11/16/2012
url = 'http://murphylab.web.cmu.edu/data/Hela/3D/multitiff';

imageDirectory = [ pwd filesep 'images' ];
if ~exist( imageDirectory, 'dir' )
    mkdir( imageDirectory );
end



if exist( imageDirectory, 'dir' ) && length( dir(imageDirectory) ) == 196
    disp('Image directory found and built');
else
    for index=1:1:length(patterns)
        pattern = patterns{index};
        disp( pattern )
        tarball = [ '3DHela_' pattern '.tgz' ];
        if ~exist( tarball, 'file' )
            disp([ 'Downloading Dataset: ' tarball ]);
            urlwrite([url filesep tarball], tarball);
            
            disp( 'Extracting files' );
            system(['tar -xvf ' pwd filesep tarball] );
        else
            disp( 'Pattern found on disk. Extracting files from existing tarball' );
            if ~exist( [pwd filesep pattern], 'file' )
                disp( 'Extracting files' );
                untar( tarball );
            else
                disp( 'Extracted folder found. Skipping extraction' );
            end
        end
    end
    
    for i=1:1:length(patterns)
        pattern = patterns{i};
        files = {'cell*ch0*.tif', 'cell*ch1*tif', 'cell*ch2*tif', '*mask*'};
        
        for j=1:1:length(files)
            filenames = ml_ls( [ pwd filesep pattern filesep files{j} ] );
            
            for k=1:1:length(filenames)
               [ path filename extension ] = fileparts( filenames{k} );
               movefile( filenames{k}, [ imageDirectory filesep pattern '_' filename  extension ] )
            end
        end
    end
end

dimensionality = '3D';
pattern = 'framework';

dna = [imageDirectory filesep '*cell*ch0*.tif'];
cell = [imageDirectory filesep '*cell*ch1*.tif'];
param.masks = [imageDirectory filesep '*mask*.tif'];

% generic model options
% ---------------------
% model.name                (optional) Holds the name of the model. Default is empty.
param.model.name = '3d_hela_framework_model';

% model.id                  (optional) Holds the id of the model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.model.id = result(1:end-1);
catch err
    param.model.id = num2str(now);
end

% model.filename            Holds the output filename.
param.model.filename = [ lower(pattern) '.xml' ];

% nuclear shape model options
% ---------------------------
% nucleus.type              Holds the nuclear model type. Default is "major axis".
param.nucleus.type = 'major axis';

% nucleus.name              (optional) Holds the name of the nuclear model. Default is empty.
param.nucleus.name = 'all';

% nucleus.id                (optional) Holds the id of the nuclear model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.nucleus.id = result(1:end-1);
catch err
    param.nucleus.id = num2str(now);
end

% cell shape model options
% ------------------------
% cell.type                 Holds the cell model type. Default is "ratio".
param.cell.type = 'ratio';

% cell.name                 (optional) Holds the name of the cell model. Default is empty.
param.cell.model = 'framework';

% cell.id                   (optional) Holds the id the cell model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.cell.id = result(1:end-1);
catch err
    param.cell.id = num2str(now);
end

% other options
% -------------
% verbose                   (optional) Displays messages to screen. The default is true.
param.verbose = true;

% debug                     (optional) Reports errors and warnings. Default is false.
param.debug = true;

% train.flag                (optional) Selects what model is going to be trained ('nuclear',
%                           'framework', or 'all'). Default is 'all'
param.train.flag = 'framework';

%this is the resolution of the datasets
param.model.resolution = [0.049, 0.049, 0.2000];

%documentation
param.documentation.author = 'Murphy Lab';
param.documentation.email = 'murphy@cmu.edu';
param.documentation.description = 'This is the framework trained on all images from the HeLa dataset.';

answer = img2slml( dimensionality, dna, cell, [], param );