function [segdna,segcell] = seg_cell_and_nucleus( imgdir, downsample, display, param )

% Author: Robert F. Murphy (murphy@cmu.edu)
%
% Copyright (C) 2012 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% March 21, 2012 R.F. Murphy Fix downsampling; add display param;
%                            Only keep biggest object in each slice
% March 22, 2012 R.F. Murphy & Devin Sullivan Find top and bottom slice
%                            separately for cell and nucleus
% March 23, 2012 I. Cao-Berg Fixed a bug when getting cellnum from imgdir
% March 25, 2012 I. Cao-Berg The method itself checks whether a temporary file exists
% March 27, 2012 I. Cao-Berg Removed top and bottom blank slices from DNA and cell images
% April 10, 2012 I. Cao-Berg Added parameter structure to hold debug and verbose flags
% July 23, 2012  R.F. Murphy Add param to enforce assumption that slice with
%                           largest area is the bottom (appropriate for adherent cells
% August 1, 2012 I. Cao-Berg Modified the code so that is saves the bottom slice index 
%                            and top slice index of the original image with 
%                            respect to segcell
% August 2, 2012 D. Sullivan Fixed a bug where the DNA top/bot where saved instead of the 
%                            cell membrane
% August 30, 2012 G. Johnson Added check_img_dirs to use pre-existing directories to 
%                            infer values for missing directories.
%                            Added a check to skip processing for any
%                            binary images.
%                            Moved all processing steps (deconvolution, cropping, etc) to
%                            a single routine for ease of upkeep.
% Sept 7, 2012 G Johnson     Added t/f flag for check_img_dirs
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

temporaryFolder = [ pwd filesep 'temp' filesep 'preprocessing' ];
if ~exist( temporaryFolder, 'dir' )
   mkdir( temporaryFolder );
end

segdna = [];
segcell = [];
if nargin == 3
  param = [];
  verbose = false;
  debug = false;
elseif nargin > 4
  warning('CellOrganizer: Wrong number of input arguments.');
  return
else
  try
     verbose = param.verbose;
     if ~islogical( verbose )
        verbose = false;
     end
  catch
     verbose = false;
  end

  try
     debug = param.debug;
     if ~islogical( debug )
        debug = false;
     end
  catch
     debug = false;
  end

  try
    adherent = param.adherent;
    if ~islogical( adherent )
        adherent = true;
    end
  catch
    adherent = true;
  end

end

if ~exist('display','var')
   display = false;
end

%icaoberg fixed bug
%gj aug 29, 2012
%changed img=='/' to img==filesep
delimiters = find( imgdir==filesep );
cellnum = str2num( imgdir(delimiters(end)+5:end) );

%icaoberg march 26, 2012
if exist( [ temporaryFolder ...
   filesep 'cell' num2str(cellnum) '.mat' ] )
   load( [ temporaryFolder ...
   filesep 'cell' num2str(cellnum) '.mat' ] );
   return
end

%gj aug 29, 2012
if ~check_img_dirs(imgdir)
    return;
end

if any(round(downsample)~=downsample)
    warning('Downsample vector rounded to integers');
end
downsample=round(downsample);

%protim3 = ml_loadimage([imgdir '/prot/'],'tif');
dnaim3 = ml_loadimage([imgdir filesep 'dna' filesep],'tif');
cellim3 = ml_loadimage([imgdir filesep 'cell' filesep],'tif');
mask = ml_loadimage([imgdir filesep 'crop' filesep],'tif');


%protim3 = tz_maskimg_3d(protim3,mask);
dnaim3 = tz_maskimg_3d(dnaim3,mask);
cellim3 = tz_maskimg_3d(cellim3,mask);
clear mask


%%%%Preprocessing starts here
if length(unique(dnaim3)) > 2
    dnafile=['dna_image' num2str(cellnum) '.mat'];
    [segdna, ~, ~, ~] = preprocess(dnaim3, dnafile, '3DHeLa_DNA_PSF.tif', downsample, false, 0.95, 0.05, display);
else
    segdna = dnaim3;
end

if length(unique(cellim3)) > 2
    cellfile=['cell_image' num2str(cellnum) '.mat'];
    [segcell, top_slice, bot_slice, sumslices] = preprocess(cellim3, cellfile, '3DHeLa_Cell_PSF.tif', downsample, adherent, 0.98, 0.02, display);
else
    segcell = cellim3;
    a = find(sum(sum(cellim3,1),2));
    bot_slice = a(1);
    top_slice = a(end);
    
    sumslices = cumsum(squeeze(sum(sum(cellim3))));
    sumslices = sumslices/sumslices(end);
end

original_segcellsize = size(segcell);
segcell = segcell(:,:,bot_slice:top_slice);
segdna = segdna(:,:,bot_slice:top_slice);

% make sure that no part of the nucleus is outside the cell
segdna = and(segdna,segcell);

if display
  try
    pause
    figure;
    for i=1:size(segdna,3)
        RGB(:,:,1)=uint8(segdna(:,:,i)*255);
        RGB(:,:,2)=uint8(segcell(:,:,i)*255);
        RGB(:,:,3)=uint8(zeros(size(segdna(:,:,i))));
        imshow(RGB); pause(0.1);
    end
  catch
  end
end

%icaoberg march 26, 2012
try
 if verbose
  disp('CellOrganizer: Saving preprocessed files to disk.');
 end

 %icaoberg 1/8/2012
 save( [ temporaryFolder ...
   filesep 'cell' num2str(cellnum) '.mat' ], 'segcell', 'segdna', 'downsample', ...
         'original_segcellsize', 'bot_slice', 'top_slice', 'sumslices' );
catch
 if verbose
   disp('CellOrganizer: Unable to save preprocessed files to disk.');
 end
end

end


%gj aug 30, 2012
function [segimg, top_slice, bot_slice, sumslices] = preprocess(img, imgFile, psfPath, downsample, adherent, top_thresh, bot_thresh, display)
    
% Load PSFs
infPSF = imfinfo( psfPath);

psf = zeros( infPSF(1).Height, infPSF(1).Width, length(infPSF));

for I=1:length(infPSF)
    psf(:,:,I)=imread(psfPath,I);
end


psf = psf.^2; % Approximate confocal PSF

% downsample PDFs and images as specified by user
if exist(imgFile,'file')
    load(imgFile)
else
    fprintf(1,'%s\n','Downsampling PSF and image'); tic;
    psf = ml_downsize(psf,downsample,'average');
    resizeimg = ml_downsize(img,downsample,'average'); toc
    clear dnaim3
    fprintf(1,'%s\n','Deconvolving image'); tic;
    [image,psf2] = deconvblind(resizeimg,psf); toc
%    save(dnafile,'dna_image','dnaPSF2');
end


fprintf(1,'%s\n','Segmenting image'); tic;
sumslices = cumsum(squeeze(sum(sum(image))));
sumslices = sumslices/sumslices(end);
bot_slice=find(sumslices> bot_thresh); %0.05, 0.02
bot_slice=bot_slice(1);
top_slice=find(sumslices< top_thresh); %0.95, 0.98
top_slice=top_slice(end);
bot_slice=max(bot_slice,bot_slice);
top_slice=min(top_slice,top_slice);

fprintf(1,'Total slices=%i, Bottom slice=%i, Top slice=%i\n',length(sumslices),bot_slice,top_slice);
segimg = active3Dsegment(image, bot_slice, top_slice,display);

if adherent %false, <param>
   [bot_slice,top_slice]=fixadherent(segimg,bot_slice,top_slice);
end
clear image
toc

    
end