function imgs = model2img( models, param )
% MODEL2IMG Synthesizes a multicolor image from a set of SLML Level 1.0
% Version 1.* instances. Saves the results to an intermedia folder and returns
% the synthesized image.
%
% List Of Input Arguments Descriptions
% ----------------------- ------------
% models                  A cell array of valid and parsed SLML instances.
% param                   A structure holding the function options
%
% The shape of param is described
%
% List Of Parameters        Descriptions
% ------------------        ------------
% targetDirectory           (optional) Directory where the images are going to be saved. Default is current directory.
% prefix                    (optional) Filename prefix for the synthesized images. Default is 'demo'
% numberOfSynthesizedImages (optional) Number of synthesized images. Default is 1.
% compression               (optional) Compression of tiff, i.e. 'none', 'lzw' and 'packbits'
% verbose                   (optional) Print the intermediate steps to screen. Default is true.
% microscope                (optional) Microscope model from which we select a point spread function. Default is 'none'
% sampling.method           (optional) Can be 'disc', 'trimmed' or 'sampled'. Default is trimmed
% sampling.density          (optional) An integer. Default is empty.
% protein.cytonuclearflag   (optional) Can 'cyto', 'nucleus' or 'all'. Default is all.
%
% Example
% -------
% instances = { 'model' };
% param.targetDirectory = pwd;
% param.prefix = 'demo'
% param.numberOfSynthesizedImages = 10;
% param.compression = 'lzw';
% param.microscope = 'svi';
% param.verbose = true;
%
% >> slml2img( instances, param );

% Author: Ivan E. Cao-Berg (icaoberg@cmu.edu)
%
% Copyright (C) 2008-2012  Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% March 7, 2012 R. F. Murphy Change scaling to 0-255
% March 8, 2012 I. Cao-Berg Documented method
% March 9, 2012 I. Cao-Berg Commented out sampling density,
% changed disp to fprintf and improved logs
% March 14, 2012 I. Cao-Berg Added protein.location parameter
% March 14, 2012 I. Cao-Berg Added contrast stretching for every pattern
% March 21, 2012 I. Cao-Berg Changed protein.location to protein.cytonuclear flag
% April 11, 2012 I. Cao-Berg Parameter structure is set to empty by default
% when the number of input arguments is one. Added try/catch clause to
% determine verbose flag. Default is false.
% April 12, 2012 R.F. Murphy Fix calculation of nucleardist and celldist;
% fix name of microtubule model
% August 4, 2012 D. Sullivan Used the param.nucleus and param.cell instead 
% of creating separate "filled" images to save memory.
% August 6, 2012 I. Cao-Berg Updated access to temporary folder so that it access the
% correct folder if more than one temporary folder is present in the current workspace path
% October 1, 2012 I. Cao-Berg Updated code so that if model2framework fails
% to make a framework and returns empty arrays, then this method returns an
% empty cell array as well as throwing a warning if the debug flag is true
% October 9, 2012 I. Cao-Berg Added synthesis option that allows user to
% synthesize and return a nuclear instance ('nucleus'), a cell instance
% ('cell'), a framework instance ('framework') or make an instance with all
% the patterns in the existing model files ('all'). Default is all
% November 14, 2012 D. Sullivan added param.resolution.cell and
% param.resolution.objects and code to adjust frameworks to proper size for
% object synthesis. 
% November 16, 2012 I. Cao-Berg Updated code so that parameter structure
% can be passed to model2framework to allow model2diffeomorphic to inherit
% parameter values from slml2img
% January 21, 2013 D. Sullivan updated resolution framework s.t. user may
% now specify multiple object model resolutions and the output will be in
% the form of the lowest resolution (highest numbers since resolutions =
% microns/pixel)
% January 22, 2013 I. Cao-Berg modified if statement so that it checks 
% whether field exists before querying it
% January 22, 2013 I. Cao-Berg extracts resolution information from first 
% model in list of models
% February 20, 2013 D. Sullivan fixed resolution extraction
% 
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

if nargin > 2
    error( 'CellOrganizer: Wrong number of input arguments.' );
end

if nargin == 1
    param = [];
end

imgs = {};

try
    fileID = param.fileID;
catch err
    fileID = [];
end

try
    sampling = param.sampling.method;
catch err
    param.sampling.method = 'disc';
end

% try
%     N = param.sampling.N;
% catch
%     param.sampling.N = [];
% end
N = [];

try
    logger = param.logger;
catch err
    logger = false;
end

try
    verbose = param.verbose;
catch err
    verbose = true;
end

%icaoberg 10/1/2012
try
    debug = param.debug;
    if ~isa( debug, 'logical' )
        debug = false;
    end
catch
    debug = false;
end

%icaoberg 10/9/2012
try
    synthesis = param.synthesis;
    if ~isa( synthesis, 'char' )
        synthesis = 'all';
    else
        if ~strcmpi( synthesis, 'nucleus' ) && ~strcmpi( synthesis, 'cell' ) && ...
                ~strcmpi( synthesis, 'framework' ) && ~strcmpi( synthesis, 'all' )
            synthesis = 'all';
        else
            synthesis = lower( synthesis );
        end
    end
catch
    synthesis = 'all';
end

%checking the validity of the input models

%initialize parameters
if isempty( models )
    error( 'CellOrganizer: Input argument models cannot be empty' );
else
    if ~isempty( fileID )
        fprintf( fileID, '%s\n', ['Setting model dimensionality to ' models{1}.dimensionality] );
    end
    
    if verbose
        fprintf( 1, '%s\n', ['Setting model dimensionality to ' models{1}.dimensionality] );
    end
    
    try
        dimensionality = models{1}.dimensionality;
    catch err
        warning('CellOrganizer: Unable to set model dimensionality');
        return
    end
end

if ~isempty( fileID )
    fprintf( fileID, '%s\n', 'Checking all models have the same dimensionality' );
end
if verbose
    fprintf( 1, '%s\n', 'Checking all models have the same dimensionality' );
end

if ~exist( [ pwd filesep 'temp'], 'dir' )
    mkdir( [pwd filesep 'temp'] );
end

for i=1:1:length(models)
    %icaoberg 10/1/2012
    if ~strcmpi( models{i}.dimensionality, dimensionality )
        warning('CellOrganizer: Models have different dimensionality. Unable to synthesize image.');
        
        if ~isempty( fileID )
            fprintf( fileID, '%s\n', 'CellOrganizer: Models have different dimensionality. Unable to synthesize image.' );
            fprintf( fileID, '%s\n', 'Closing log file' );
            fclose( fileID );
        end
        
        return;
    end
end

switch lower(dimensionality)
    %synthesizes 2D images
    case '2d'
        if ~exist( 'param', 'var' )
            param  = [];
        end
        
        imgs = [];
        %the size of the images is fixed to be 1024x1024
        param = ml_initparam(param, ...
            struct('imageSize',[1024 1024],'gentex',0,'loc','all'));
        [nucEdge,cellEdge,outres] = model2framework( models{1}, param );
        param.resolution.cell = outres;

        %icaoberg 10/1/2012
        if isempty( nucEdge ) || isempty( cellEdge )
            if debug
                warning('Unable to synthesize 2D framework');
                imgs = {};
                return
            end
        end
            
        %post-process each channel
        se=strel('disk',4,4);
        cellimage = imdilate(cellEdge,se);
        if ~exist('nuctex','var')
            nucimage = imdilate(nucEdge,se);
        else
            nucimage = nucEdge;
        end
        
        %icaoberg 10/9/2012
        %check whether the synthesis flag is nuclear, cell, framework
        if strcmpi( synthesis, 'nucleus' ) || strcmpi( synthesis, 'cell' ) ...
                || strcmpi( synthesis, 'framework' )
            imgs(:,:,1) = ml_bcimg(double(nucimage),[],[0 255]);
            imgs(:,:,2) = ml_bcimg(double(cellimage),[],[0 255]);
            return
        end
        
        for j=1:1:length(models)
            protimage{j} = ...
                ml_genprotimg2D( models{j}.proteinModel, ...
                nucEdge, cellEdge, param );
        end
        
        %post-process each channel
        se=strel('disk',4,4);
        cellimage = imdilate(cellEdge,se);
        if ~exist('nuctex','var')
            nucimage = imdilate(nucEdge,se);
        else
            nucimage = nucEdge;
        end
        
        %add channels into a single image
        imgs(:,:,1) = ml_bcimg(double(nucimage),[],[0 255]);
        imgs(:,:,2) = ml_bcimg(double(cellimage),[],[0 255]);
        for j=1:1:length(protimage)
            imgs(:,:,2+j) = ml_bcimg(double(protimage{j}),[],[0 255]);
        end
        save( ['temp' filesep 'image.mat'], 'imgs' );
    case '3d'
        %synthesize 3D images
        vesicle = {};
        for j=1:1:length(models)
            if strcmpi( models{i}.proteinShape.type, 'vesicle' )
                vesicle{length(vesicle)+1} = j;
            end
        end
        
        if ~isempty( vesicle )
            fprintf( 1, '%s\n', ['At least one vesicle model was found, looking for a cell shape ' ...
                ' and a nuclear shape model.']);
            if isfield( models{1}, 'nuclearShapeModel' ) && isfield( models{1}, 'cellShapeModel' )
                fprintf( 1, '%s\n', 'A nuclear shape and cell shape was found in the first model.');
            end
        end
        
        microtubules = {};
        for j=1:1:length(models)
            if strcmpi( models{j}.proteinShape.type, 'network' ) && ...
                    strcmpi( models{j}.proteinShape.class, 'microtubule' )
                microtubules{length(microtubules)+1} = j;
            end
        end
        
        if ~isempty( microtubules )
            fprintf( 1, '%s\n', ['At least one microtubule model was found, ' ...
                'looking for a centrosome model.'] );
            centrosome = {};
            for j=1:1:length(models)
                if strcmpi( models{j}.proteinShape.type, 'centrosome' ) && ...
                        strcmpi( models{j}.proteinShape.class, 'centrosome' )
                    centrosome{length(centrosome)+1} = j;
                end
            end
        end
        
        if ~isempty( microtubules ) && isempty( centrosome )
            if ~isempty( fileID )
                fprintf( fileID, '%s\n', ['At least one microtubule model was found ' ...
                    'in your list but no centrosome models were found.'] );
                fprintf( fileID, '%s\n', 'Closing log file' );
                fclose( fileID );
            end
            
            error( ['CellOrganizer: ' num2str(length(microtubules)) ' microtubule models where found ' ...
                'in your list but no centrosome models found.'] ) %uncool
        else
            if exist( 'centrosome', 'var' )
                fprintf( 1, '%s\n', ['Centrosome models found:' num2str(length(centrosome))] );
            end
            
            %the cell framework, that is the nuclear and cell shape, are
            %generated using the first model found in the list
            fprintf( 1, '%s\n', 'Generating cell framework' );
            %icaoberg 11/16/2012
            
            %D. Sullivan 2/24/13 moved resolution setup to above
            %Note, setting initial framework resolution off first object
            %model
            %model2framework
            %Framework(cell)
            if isfield(models{1}.cellShapeModel,'resolution')
                
                
                %D. Sullivan 3/6/13 check if a framework instance was
                %passed and if so use that resolution. If none was
                %specified, return an error and exit
                if isfield(param,'instance')
                    if isfield(param.instance,'cell')
                        if isfield(param,'resolution')
                            
                            if isfield(param.resolution,'cell')
                                
                            else
                                warning('CellOrganizer error: no resolution specified for the pre-segmented framework (param.resolution.cell)');
                                error('Cannot synthesize images');
                            end
                        else
                            warning('CellOrganizer error: no resolution specified for the pre-segmented framework (param.resolution.cell)');
                            error('Cannot synthesize images');
                        end
                    else
                        param.resolution.cell = models{1}.cellShapeModel.resolution;
                    end
                else
                    param.resolution.cell = models{1}.cellShapeModel.resolution;
                end
                
            elseif isfield(param.resolution,'cell')
                
            else
                error('No cell resolution specified');
            end
            
            %objects
            if isfield(models{1}.proteinShape,'resolution')
                param.resolution.objects = models{1}.proteinShape.resolution;
            elseif isfield(param.resolution.objects)

            else
                error('No object resolution specified');
            end

            [param.nucleus, param.cell,outres] = model2framework( models{1}, param );
            param.resolution.cell = outres;
            
            %icaoberg 10/1/2012
            if isempty( param.nucleus ) || isempty( param.cell )
                if debug
                    warning( 'Unable to synthesize 3D framework' );
                end
                
                imgs = {};
                return
            end
            
            fprintf( 1, '%s\n', 'Filling cell and nuclear images' );
           
            %devins 8/4/12
            %used the param.nucleus and param.cell instead of creating
            %seperate "filled" images to save memory.
            
            %param.nucleusfilled=imfill(param.nucleus>0,'holes');
            %param.cellfilled=imfill(param.cell>0,'holes');
            
            param.nucleus=imfill(param.nucleus>0,'holes');
            param.cell=imfill(param.cell>0,'holes');
            
            %D. Sullivan 11/14/12
            %moved resolution code from the model2instance in case the user
            %trained all object models at a single resolution
            %**this also allows the distance images to be calculated after
            %the resolution adjustment**
            %Check if the protein models were learned at a different
            %resolution than the cell shape models 
            %D. Sullivan 1/21/13 moved above cell/nuc image save so the
            %images are saved at the correct resolution
            
            try
                %icaoberg 1/22/13
                %extracts resolution information from first model in list
                %D. Sullivan 2/24/13, change priority of resolution to prefer
                %model associated resolution
                %D. Sullivan 2/21/13, First try the user specified param struct.
%                 if isfield(models{1}.cellShapeModel,'resolution')
%                     param.resolution.cell = models{1}.cellShapeModel.resolution;
%                 elseif isfield(param.resolution,'cell')
%                        
%                 else 
%                     error('No cell resolution specified!');
%                 end
%                 
                %D. Sullivan 2/20/13
                %loop through and get resolution for each object model
                if isfield(models{1}.proteinShape,'resolution')
                    param.resolution.objects = zeros(length(models),3);
                    for i = 1:length(models)
                        param.resolution.objects(i,:) = models{1}.proteinShape.resolution;
                    end
                elseif isfield(param.resolution,'objects')
                    
                else
                    error('No object resolutions specified!');
                end
                
                if( size( param.resolution.objects, 1 ) == 1 )
                    %since the cell is built conditionally on the nucleus, they should
                    %ALWAYS have the same resolution
                    initres = param.resolution.cell;%e.g. [0.05,0.05,0.35]
                    finalres = param.resolution.objects;
                    param.outputres=finalres;
                    
                    %nucleus
                    finalsize_x = floor(initres(1)./finalres(1).*size(param.nucleus,1));
                    finalsize_y = floor(initres(2)./finalres(2).*size(param.nucleus,2));
                    finalsize_z = floor(initres(3)./finalres(3).*size(param.nucleus,3));
                    param.nucleus = imresize(param.nucleus,[finalsize_x finalsize_y],'bilinear');
                    
                    %need to resize the z
                    param.nucleus = tp_stretch3d(param.nucleus,finalsize_z);
                    param.nucleus = param.nucleus>0;
                    
                    %cell
                    %Note: the recalculation of final size should be unnecessary since the
                    %cell and nucleus images should always be the same size, but since the
                    %arithmatic is trivially fast I recalculate to deal with potential
                    %weird situations in the future(e.g. if we need the nuclear image to be
                    %a smaller object that we add in to the cell image for space)DPS
                    finalsize_x = floor(initres(1)./finalres(1).*size(param.cell,1));
                    finalsize_y = floor(initres(2)./finalres(2).*size(param.cell,2));
                    finalsize_z = floor(initres(3)./finalres(3).*size(param.cell,3));
                    param.cell = imresize(param.cell,[finalsize_x finalsize_y],'bilinear');
                    
                    %need to resize the z
                    param.cell = tp_stretch3d(param.cell,finalsize_z);
                    param.cell = param.cell>0;
                elseif(size(param.resolution.objects,1)>1)
                    param.outputres = max(param.resolution.objects);
                end
            catch
                warning(['CellOrganizer: No resolution specified for either cell or object class ',...
                    'assuming no resizing necessary. If this is incorrect unexpected results will occur'])
            end
            
            
            %icaoberg 10/9/2012
            %save the synthetic nucleus to temporary folder
            if strcmpi( synthesis, 'nucleus' ) || ...
                    strcmpi( synthesis, 'framework' ) || ...
                    strcmpi( synthesis, 'all' )
                img = param.nucleus;
                %icaoberg 8/6/2012
                save( [pwd filesep 'temp/image1.mat'], 'img' );
                imgs{length(imgs)+1} = img;
                clear img;
            end
            
            %if the user only requested a nucleus, stop method
            if strcmpi( synthesis, 'nucleus' )
                return
            end
            
            %save synthetic cell to temporary folder
            if strcmpi( synthesis, 'cell' ) || ...
                    strcmpi( synthesis, 'framework' ) || ...
                    strcmpi( synthesis, 'all' )
                img = param.cell;
                %icaoberg 8/6/2012
                save( [pwd filesep 'temp/image2.mat'], 'img' );
                imgs{length(imgs)+1} = img;
                clear img;
            end
            
            %if the user requested a cell membrane or a framework, stop
            %method
            if strcmpi( synthesis, 'cell' ) || strcmpi( synthesis, 'framework' )
                return
            end
            
            fprintf( 1, '%s\n', ['Computing Euclidean distance transform to cell and nuclear edges'] );
            param.celldist = bwdist(bwperim(param.cell));
            param.nucleardist = bwdist(bwperim(param.nucleus));
            param.nucleardist(param.nucleus==1) = -param.nucleardist(param.nucleus==1);
            
            if exist( 'centrosome', 'var' ) && ~isempty( centrosome )
                fprintf( 1, '%s\n', 'Generating centrosome.' );
                [param.centrosome,outres] = model2instance( models{centrosome{1}}.proteinShape, param );
            end
            
            if verbose
                fprintf( 1, '%s\n', ['Saving temporary cell shape model instance'] );
            end
            
            clear blank;
            clear box;
            clear instance;
            clear nuclei;
            
            for j=1:1:length(models)
                param.currentmodelnum = j;
                if ~strcmpi( models{j}.proteinShape.class, 'centrosome' )
                    if verbose
                        tic
                        disp( ['Generating protein pattern model instance ' num2str(j) ' from ' models{j}.proteinShape.class ' pattern'] );
                    end
                    
                    [img,outres] = model2instance( models{j}.proteinShape, param );
                    %img = ml_bcimg(double(img),[],[]);
                    
                    if verbose
                        disp( ['Saving temporary protein model instance ' num2str(j+2) ] );
                    end
                    %icaoberg 8/6/2012
                    save( [pwd filesep 'temp' filesep 'image' num2str(j+2) '.mat'], 'img' );
                    imgs{length(imgs)+1} = img;
                    clear img;
                end
            end
            
            %D.Sullivan 1/21/13 readjust cell and nucleus to be same
            %resoltuion as final object models and resave temporary files
            if( isfield( param, 'resolution' ) && size(param.resolution.objects,1)>1)
                %since the cell is built conditionally on the nucleus, they should
                %ALWAYS have the same resolution
                initres = param.resolution.cell;%e.g. [0.05,0.05,0.35]
                finalres = param.outputres;
                
                %nucleus
                finalsize_x = floor(initres(1)./finalres(1).*size(param.nucleus,1));
                finalsize_y = floor(initres(2)./finalres(2).*size(param.nucleus,2));
                finalsize_z = floor(initres(3)./finalres(3).*size(param.nucleus,3));
                param.nucleus = imresize(param.nucleus,[finalsize_x finalsize_y],'bilinear');
                
                
                %need to resize the z
                param.nucleus = tp_stretch3d(param.nucleus,finalsize_z);
                param.nucleus = param.nucleus>0;
                imgs{1} = param.nucleus;
                %resave the synthetic nucleus to temporary folder
                if strcmpi( synthesis, 'nucleus' ) || ...
                        strcmpi( synthesis, 'framework' ) || ...
                        strcmpi( synthesis, 'all' )
                    img = param.nucleus;
                    %icaoberg 8/6/2012
                    save( [pwd filesep 'temp/image1.mat'], 'img' );
                    clear img;
                end
                
                %cell
                %Note: the recalculation of final size should be unnecessary since the
                %cell and nucleus images should always be the same size, but since the
                %arithmatic is trivially fast I recalculate to deal with potential
                %weird situations in the future(e.g. if we need the nuclear image to be
                %a smaller object that we add in to the cell image for space) DPS
                finalsize_x = floor(initres(1)./finalres(1).*size(param.cell,1));
                finalsize_y = floor(initres(2)./finalres(2).*size(param.cell,2));
                finalsize_z = floor(initres(3)./finalres(3).*size(param.cell,3));
                param.cell = imresize(param.cell,[finalsize_x finalsize_y],'bilinear');
                
                %need to resize the z
                param.cell = tp_stretch3d(param.cell,finalsize_z);
                param.cell = param.cell;
                param.cell = param.cell>0;
                imgs{2} = param.cell;
                %resave synthetic cell to temporary folder
                if strcmpi( synthesis, 'cell' ) || ...
                        strcmpi( synthesis, 'framework' ) || ...
                        strcmpi( synthesis, 'all' )
                    img = param.cell;
                    %icaoberg 8/6/2012
                    save( [pwd filesep 'temp/image2.mat'], 'img' );
                    clear img;
                end
                 
            end  
            
        end
    otherwise
        error( 'CellOrganizer: Unknown dimensionality or dimensionality argument missing.' );
end
end%model2img
