function [nucimg, cellimg, outres] = model2framework( model, param )
%MODEL2FRAMEWORK Helper method that generates a 2D/3D framework from a
%valid SLML model. The models that are currently supported are
% (1) 2D spline model of the nucleus + 2D ratio model of the cell membrane
% (2) 3D spline model of the nucleus + 3D ratio model of the cell membrane
% (3) 3D diffeomorphic model of the nucleus and cell membrane

% Author: Ivan E. Cao-Berg (icaoberg@scs.cmu.edu)
%
% Copyright (C) 2012 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% April 7, 2012 R.F. Murphy Use modified function for synthesizing cell
% shape; adding additional models for height ratios and nuclear position
% September 28, 2012 I. Cao-Berg Added documentation and checking of input
% arguments
% October 1, 2012 I. Cao-Berg Added diffeomorphic model structure so that
% the method knows what to do when a model of this type is added. I also
% added a default value so that if the model type is not recognized, the
% method returns empty nuclear and cell images
% October 1, 2012 I. Cao-Berg Added helper method that synthesizes an
% instance from a diffeomorphic model
% November 1, 2012 I. Cao-Berg Added option to parameter structure that allows
% the selection between two methods when synthesizing instances from diffeomorphic
% models. This option is meant for testing, and it is not documented in top 
% level methods
% November 5, 2012 I. Cao-Berg Fixed bug in method that synthesizes images from 
% diffeomorphic models that was returning an empty image due to wrong index
% November 6, 2012 I. Cao-Berg Updated synthesis code to use Euler integrator
% January 20, 2013 T. Buck Updated method to match new options from newly
% trained diffeomorphic model
% February 4, 2013 D. Sullivan made file return a resolution associated
% with the cell and nuclear shapes. If there is none specified, prints a
% warning and returns the outres field as blank. 
% February 22, 2013 D. Sullivan made the nuclear shape synthesis dependent
%                               on the object model resolution
% February 24, 2013 D. Sullivan Moved resolution checks to model2img
% March 3, 2013 D. Sullivan Added resolution outres to the 2D synthesis
% March 5, 2013 D. Sullivan Added check for an existing framework
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

    
if nargin > 2
    error('CellOrganizer: Wrong number of input arguments.' );
end

if length(model) > 1
    %icaoberg 9/28/2012
    warning(['CellOrganizer: More than one model has been found. Synthesizing framework ' ...
        ' from the first model in the cell array']);
    model = model{1};
end

if ~exist( 'param', 'var' )
    param = [];
end

%icaoberg 10/1/2012
try
    verbose = param.verbose;
    if isa( verbose, 'logical' )
        verbose = false;
    end
catch
    verbose = false;
end

%icaoberg 9/28/2012
try
    debug = param.debug;
    if isa( debug, 'logical' )
        debug = false;
    end
catch
    debug = false;
end

try
    dimensionality = model.dimensionality;
catch
    %icaoberg 9/28/2012
    error( 'CellOrganizer: Unable to set dimensionality' );
    nucimg = [];
    cellimg = [];
    return
end

switch lower(dimensionality)
    case '2d'
        %generate 2D framework
        param = ml_initparam(param, ...
            struct('imageSize',[1024 1024],'gentex',0,'loc','all'));
        
        %generate cell framework
        [nucimg,cellimg] = ml_gencellcomp2D( model, param );
        
        %D. Sullivan 3/3/12
        %Set the output resolution
        if isfield(param,'resolution')
            if isfield(param.resolution,'objects')
                outres = param.resolution.objects;
            end
            outres = param.resolution;
        else
            outres = [];
        end
    case '3d'
        %D. Sullivan 3/5/13 
        %check if we already have a framework (as in from raw data)

        if isfield( param, 'instance' ) && isfield(param.instance,'nucleus')
            if isfield(param.instance,'cell')
                %check if the images are the same size in all dimensions
                if all(size(param.instance.cell)==size(param.instance.nucleus))
                    nucimg = param.instance.nucleus;
                    cellimg = param.instance.cell;
                    if isfield(param,'resolution')
                        if isfield(param.resolution,'cell')
                            outres = param.resolution.cell;
                            
                        end
                    else
                        outres = [];
                    end
                    return 
                    
                end
                if debug
                    warning('Cell and nucleus found, but the images were not the same size, synthesizing new framework');
                end
            end
        end
        
        if strcmpi( model.nuclearShapeModel.type, 'medial axis' ) && ...
                strcmpi( model.cellShapeModel.type, 'ratio' )
            
            %generate 3D framework
            model.nuclearShapeModel.height.stat.mu = model.nuclearShapeModel.height.stat.mu;
            model.nuclearShapeModel.height.stat.sigma = model.nuclearShapeModel.height.stat.sigma;
            
            %D. Sullivan 2/22/13 added param structure to pass the cell and
            %object resolutions 
            instance = tp_genspsurf(model.nuclearShapeModel,param);
            %instance = tp_genspsurf(model.nuclearShapeModel);
            
            %disp('Generating nuclear shape instance');
            [nucimg,nucsurf,outres] = tp_gennucshape(instance,param);
            
            nucleus.nucimgsize = size(nucimg);
            nucleus.nucsurf = nucsurf;
            nucleus.nucimg = nucimg;
            
            %disp('Generating cell shape instance');
            [cellimg,nucimg] = ml_gencellshape3d( ...
                model.cellShapeModel, nucleus,param );
            
            box = tp_imbox(cellimg);
            nucimg = nucimg(box(1):box(2),box(3):box(4),:);
            cellimg = cellimg(box(1):box(2),box(3):box(4),:);
            
        elseif strcmpi( model.nuclearShapeModel.type, 'diffeomorphic' ) && ...
                strcmpi( model.cellShapeModel.type, 'diffeomorphic' )
            %icaoberg 10/1/2012
            outres = param.resolution.cell;
            [nucimg, cellimg] = model2diffeomorphicInstance( model, param );
        else
            %icaoberg 10/1/2012
            warning( 'CellOrganizer: Unrecognized model type or combination of model types.' );
            nucimg = [];
            cellimg = [];
        end
    otherwise
        return
end
end

function [nucimg, cellimg] = model2diffeomorphicInstance( model, param )
%MODEL2DIFFEORMORPHICINSTANCE Helper method that synthesizes a framework
%from a diffeomorphic model
%icaoberg 10/1/2012

try
    verbose = param.verbose;
    if isa( verbose, 'logical' )
        verbose = false;
    end
catch
    verbose = false;
end

%icaoberg 9/28/2012
try
    debug = param.debug;
    if isa( debug, 'logical' )
        debug = false;
    end
catch
    debug = false;
end

try
    %icaoberg 10/1/2012
    %number_shapes = size(y2, 1);
    number_shapes = size(model.nuclearShapeModel.positions, 1);
    
    % Generate a random point inside the convex hull of training shapes in the
    % shape space (probably not uniform sampling):
    random_weights = rand(number_shapes, 1);
    random_weights = random_weights ./ sum(random_weights);
    
    %icaoberg 10/1/2012
    %random_point = random_weights' * y2;
    random_point = random_weights' * model.nuclearShapeModel.positions;
    
    % Generate the shape associated with that point:
    options = struct();
    options.use_compression = false;

    %icaoberg 11/1/2012
    try
       options.method = param.synthesis.diffeomorphic.method;
    catch
       %icaoberg
       %options.method = 'convnfft';
       options.method = 'convnfft_fast';
    end

    %yy 11/28/2012
    try
       options.convergence_absolute_error = param.synthesis.diffeomorphic.convergence_absolute_error;
    catch

    end

    %icaoberg 11/9/2012
    %maximum number of iterations
    try
       options.maximum_iterations = param.synthesis.diffeomorphic.maximum_iterations;
    catch
       options.maximum_iterations = 100;
    end

    %changes per step
    try
       options.convergence_tolerance = param.synthesis.diffeomorphic.convergence_tolerance;
    catch
       options.convergence_tolerance  = 0;
       %options.convergence_tolerance  = 8.279212e-02;
    end

    %changes per step
    try
       options.convergence_registration_error_scale = param.synthesis.diffeomorphic.convergence_registration_error_scale;
    catch
       options.convergence_registration_error_scale  = 5e-3;
    end

    %tebuck 11/5/2012
    %forward euler method as rk integrator
    %options.integrator_c = 0;
    %options.integrator_b = 1;
    %options.integrator_a = 0;
    %options.integrator_b2 = [];
    %options.integrator_first_same_as_last = false;    
    
    %tebuck 11/9/2012
    options.alpha = 1;          % coefficient of Lapacian operator 
    options.gamma = 0.04;       % Coefficient of the Identity 
    options.keep_proportion = 0; 
    options.single_sided = false; 
    options.max_time = 50000; 
    options.step_size = 0.01; 
    options.use_fft = true; 
    options.use_gaussian_kernel = false; 
    options.absolute_distance_tolerance = 0; 
    options.absolute_deformation_tolerance = 0; 
    options.periodic_space = false; 
    options.use_compression = false; 
    options.drop_kernel = true; 
    options.maximum_registration_error_failures = 0;

    %tebuck 11/25/2012
    %note that the following code is intended for use with values of 1 or 2
    %downsampling scale
    try
      options.downsampling_scale = param.synthesis.diffeomorphic.downsampling_scale;
    catch
      options.downsampling_scale = 1;
    end

    options.maximum_deformation_per_step = [4/options.downsampling_scale, 4/options.downsampling_scale, 0.5];
    
    %tebuck updated radius to match new model
    options.window_radius = 192/options.downsampling_scale; 
    options.filter_radius = 16/options.downsampling_scale;
    
    %icaoberg 10/1/2012
    frame = generate_frame_from_trained_shape_space_model(model, random_point, options );
    
    %icaoberg 10/8/2012
    %imwrite(reshape_contrast(round(frame.interpolated_image.get_data())), image_filename)
    img = reshape_contrast(round(frame.interpolated_image.get_data()));
    img = reshape( img, size(img,1), size(img,1), [] );

    %tebuck 11/5/2012
    %the variable image must contain three unique values: 0 for background, 1 for cell and
    %2 for nucleus, i.e. img is an indexed image    

    values = unique( img );    
    nucimg = zeros( size(img) );
    nucimg(find(img == values(3))) = 255;
  
    cellimg = zeros( size(img) );
    %icaoberg 11/5/2012
    cellimg(find(img == values(2))) = 255 ;
    
    %D. Sullivan 2/25/13 need to remove the padding zeros from the
    %framework model maintaining a 1 slice padding (this is for converting
    %to meshing if needed later. 
    keepslices = find(sum(sum(cellimg,1),2)~=0);
    cellimg = cellimg(:,:,keepslices(1)-1:keepslices(end)+1);
    nucimg = nucimg(:,:,keepslices(1)-1:keepslices(end)+1);
    
    
catch err
    getReport( err, 'extended')
    nucimg = [];
    cellimg = [];
end
end
