function model = img2model( dimensionality, varargin )
%IMG2MODEL Trains a generative model of protein subcellular location from a
%collection of microscope images.
%
%Inputs
%dimensionality    either '2D' or '3D'
%param             a structure holding possible parameter options
%
%See also IMG2SLML

% Author: Ivan E. Cao-Berg
% Copyright (C) 2007-2012 Murphy Lab
% Carnegie Mellon University
%
% ?? ??, 2011 I. Cao-Berg Added 3D model training functionality
% March 23, 2012 I. Cao-Berg Added the creation/deletion of temporary folder
% March 28, 2012 I. Cao-Berg Added a control structure under which if one or more of the 
%                image folders are nonexistent or they do not contain images,
%                the method exits
% March 28, 2012 I. Cao-Berg Added verification of input arguments when training a 2D
%                            generative model
% March 28, 2012 I. Cao-Berg Added verification of input arguments when training a 3D
%                            generative model
% April 10, 2012 I. Cao-Berg Added debug flag to the method. If flag is true, temporary
%                            files will not be deleted
% April 11, 2012 I. Cao-Berg Added verbose flag to the method
% April 17, 2012 I. Cao-Berg Returns an empty model when model cannot be trained
% July 5, 2012 I. Cao-Berg Added training flags to method so that users can train whatever component they wish
% July 26, 2012 Y.Yu Fixed a bug of the order of input argument for ml_traingenmodel2D method
% August 29, 2012 G. Johnson Modified method call to include parameter structure
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu/ or
% send email to murphy@cmu.edu

% Get collection of images
nuclearModelImages = [];
cellModelImages = [];
proteinModelImages = [];

model = [];
switch lower(dimensionality)
    case '2d'
        %icaoberg march 28, 2012
        %check number of input arguments. if there are not 5 input
        %arguments, then the method returns an empty model
        if nargin ~= 5
           warning('CellOrganizer: Wrong number of input arguments.');
           return;
        end

        %check the image directories. if they don't exist or they are
        %empty, the method returns an empty model
        nuclearModelImagesDirectory = varargin{1};
        if ~isa( nuclearModelImagesDirectory, 'char' )
           warning('CellOrganizer: DNA images directory does not exist.');
           return;
        end

        cellModelImagesDirectory = varargin{2};
        if ~isa( cellModelImagesDirectory, 'char' )
           warning('CellOrganizer: Cell images directory does not exist.');
           return;
        end
 
        proteinModelImagesDirectory = varargin{3};
        if ~isa( proteinModelImagesDirectory, 'char' )
           warning('CellOrganizer: Protein images directory does not exist.');
           return;
        end
    
        nuclearModelImages = getImages( ...
            nuclearModelImagesDirectory );
        cellModelImages = getImages( ...
            cellModelImagesDirectory );
        proteinModelImages = getImages( ...
            proteinModelImagesDirectory );
        
        if isempty( nuclearModelImages ) || ...
           isempty( cellModelImages ) || ...
           isempty( proteinModelImages )
           warning(['CellOrganizer: One or more directories are nonexistent ' ...
             'or they do not contain images.']);
          return
        end

        %check that the input parameter is a structure
        param = varargin{4};
        if ~isa( param, 'struct' )
           warning('CellOrganizer: Input argument parameter must be a struct.');
           return;
        end

        %DEBUG MODE!
        try
            debug = param.debug;
            if ~isa( debug, 'logical' )
                debug = false;
            end
        catch
            debug = false;
        end

        try
           verbose = param.verbose;
           if ~isa( debug, 'logical' )
               verbose = false;
           end
        catch
           debug = false;
        end        
        
        %check existence of temporary folder and make on if it doesn't exist
        if ~exist([pwd filesep 'temp' filesep 'preprocessed'])
          mkdir([pwd filesep 'temp' filesep 'preprocessed']);
        end
        
        % train generative model of protein subcellular location
        param.disp = 'false';
        model = ml_traingenmodel2D( ...
            proteinModelImages,...
            nuclearModelImages, ...
            cellModelImages, ...
            [], param );
    case '3d'
        %icaoberg march 28, 2012
        %check the existence of the image directory
        imgDirectory = varargin{1};
        if ~exist( imgDirectory )
           warning('CellOrganizer: Image directory does not exist.');
           return
        end

        param = varargin{2};
        if ~isa( param , 'struct' )
            warning('CellOrganizer: Input parameter list must be a structure.');
            return
        end

        try
           debug = param.debug;
           if ~isa( debug, 'logical' )
               debug = false;
           end
        catch
           debug = false;
        end

        try
           verbose = param.verbose;
           if ~isa( debug, 'logical' )
               verbose = false;
           end
        catch
           debug = false;
        end
        
        %mmackie july 3 2012
        try
            trainFlag = param.train.flag;
            if ~isa( trainFlag, 'char' );
                error('CellOrganizer: training flag must be a string');
            end
        catch
            param.train.flag = 'all';
            trainFlag = param.train.flag;
        end
        
        %icaoberg july, 5 2012
        if ~strcmpi( trainFlag, 'nuclear' ) && ...
           ~strcmpi( trainFlag, 'framework' ) && ...
           ~strcmpi( trainFlag, 'all' )
            error('CellOrganizer: unrecognized training flag');
        end

        %check existence of temporary folder and make on if it doesn't exist
        if ~exist( [ pwd filesep 'temp'], 'dir' )
            mkdir( [ pwd filesep 'temp'] );
        end
        
        try
            %icaoberg april 17, 2012
            model.dimensionality = '3D';
            
            if verbose
               %clc; 
               fprintf( 1, '%s\n', 'Training nuclear shape model' );
            end
                
            %gj aug 29, 2012
            %passes in 'param' now
            model.nuclearShapeModel = train_nuc_shape_model( ...
                imgDirectory, param );

            %mmackie july 3, 2012
            if strcmpi( trainFlag, 'framework' ) || strcmpi( trainFlag, 'all' )
                if verbose
                  %clc; 
                  fprintf( 1, '%s\n', 'Training cell shape model' );
                end
                model.cellShapeModel = train_cell_shape_model2( ...
                    imgDirectory );
            end

            if strcmpi( trainFlag, 'all' )
                if verbose
                  %clc; 
                  fprintf( 1, '%s\n', 'Training protein model' );
                end
                %D. Sullivan 2/22/13 added param structure to pass
                %resolution
                model.proteinShape = train_protein_model( ...
                   imgDirectory,param );
%                 model.proteinShape = train_protein_model( ...
%                    imgDirectory );
                %clc
            end

           %icaoberg 22/02/2013
           model.info.original_resolution = param.model.original_resolution;
           model.info.downsampling_vector = param.model.downsampling;
           model.nuclearShapeModel.resoluton = param.model.resolution;
           model.cellShapeModel.resolution = param.model.resolution;
           %D. Sullivan 2/24/2013 This should be set already in
           %train_protein_model
           %model.proteinShape.resolution = param.model.protein_resolution;

        catch err
           %icaoberg april 17, 2012
           %returns empty model if model cannot be trained
                     
           model = [];
           warning('CellOrganizer: Unable to train 3D generative model.');
          
           %icaoberg 06/02/2013
           if debug
               getReport( err, 'extended' )
           end
           
           if ~debug
              rmdir( 'temp', 's' );
           end
           
           return
        end
    otherwise
        error(['CellOrganizer: Unknown dimensionality ' ...
            dimensionality]);
end

if ~debug
   rmdir( 'temp', 's' );
else
   if verbose
       fprintf(1, '%s\n', ...
         'CellOrganizer: Debug mode is on, temporary files will not be deleted.');
   end
end

end%img2model
