function find_cell_codes( impath, savepath, param )
%FIND_CELL_CODES Calculates cell codes

% Author: Tao Peng
% March 12, 2012 Added another preprocessing routine. This routine mimics
%                the preprocessing from A. Shariff, G. K. Rohde and R. F. Murphy (2010) A
%                Generative Model of Microtubule Distributions, and Indirect Estimation of
%                its Parameters from Fluorescence Microscopy Images. Cytometry Part A 77A:457-466.
% March 19, 2012 I. Cao-Berg Added Active Countour 3D segmentation
% March 22, 2012 I. Cao-Berg Added param.downsample so that user can select a
%                downsampling scale. The default is [1 1 1] which is
%                the one in the original generative model paper
% March 23, 2012 I. Cao-Berg Fixed a bug when getting cellnum from impath
% March 27, 2012 I. Cao-Berg When calling ml_parsecell instead of sending a slice
%                of the nucleus, now we are sending a z-projection of the nucleus
% April 5, 2012 I. Cao-Berg Added new preprocessing method
% April 9, 2012 R.F. Murphy Save cell-nucleus height ratio and nuc bottomslice
% June 19, 2012 I. Cao-Berg Changed index name in the preprocessing for-loops that had the same
%                           index name as the main loop
% July 26, 2012 I. Cao-Berg If the number of slices in the nuclear image that have
%                           fluorescence is less than 4, then ignore such image
%
% Copyright (C) 2011-2012 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

if nargin == 2
   param = [];
end

try
   downsample = param.downsample;
catch
   downsample = [5 5 1];
end

%%%%celldirlist = ml_dir([impath '/cell*']);
preprocessingFolder = [ pwd filesep 'temp' filesep 'preprocessing' ];
if ~exist( preprocessingFolder )
   mkdir( preprocessingFolder );
end
celldirlist = ml_dir([preprocessingFolder '/cell*.mat']); %%%%

for i = 1:length(celldirlist)
    temporaryResultsFile = [savepath filesep 'cellcodes_' int2str(i) '.mat'];
    nucbody = {}; cellbody = {};
    disp( ['Image:' num2str(i)]);
    if ~exist( temporaryResultsFile, 'file' )
      try    
        %icaoberg april 5, 2012
        %check if preprocessing results exist
%%%%        cellfile = [ preprocessingFolder filesep 'cell' celldirlist{i}(5:end) '.mat' ];
        cellfile = [ preprocessingFolder filesep 'cell' celldirlist{i}(5:end) ]; %%%%
        if ~exist( cellfile, 'file' )
           [segdna,segcell] = seg_cell_and_nucleus([impath filesep ...
                celldirlist{i}], downsample );
           save( cellfile, 'segdna', 'segcell' );
        else
           disp('Preprocessed results found.');
           load( cellfile );
        end

        %icaoberg 26/7/2012
        nnzpixel = squeeze(sum(sum(segdna)));
  
        if length(find(nnzpixel)>0) < 4
          disp(['Ignoring image ' num2str(i)]);
          continue
        end

        %icaoberg april 5, 2012
        stacknumber = 15;
        [segdna, segcell] = ml_rescaleImage2Cell( segdna, segcell, stacknumber );        

        %fills the empty slices of DNA with the closest
        z = find(sum(sum(segdna,1),2));
        segdna(:,:,1:min(z)-1)=repmat(segdna(:,:,min(z)),[1,1,min(z)-1]);
        segdna(:,:,max(z)+1:end)=repmat(segdna(:,:,max(z)),[1,1,size(segdna,3)-max(z)]);

        %needed for model of nuclear position
        cellnucheightratio = stacknumber/length(z);
        nucbottomslice = (min(z)-1)/stacknumber;

        %preprocess DNA image
        se = strel('disk',3);
        crossectionArea = [];
        %icaoberg june 19, 2012 changed index from i to j
        for j=1:1:size(segdna,3)
            mask = segdna(:,:,j);
            mask = imclose(mask,se);
            mask = imfill(mask,'holes');
            obj = ml_findmainobj2d_bw(mask);
            crossectionArea(j) = nnz(mask);
            nucbody{j}=obj;
        end
        [maxArea,equatorZ] = max(crossectionArea);

        %preprocess cell image
        %icaoberg june 19, 2012 changed index from i to j
        for j=1:1:size(segcell,3)
            mask = segcell(:,:,j);
            mask = imclose(mask,se);
            mask = imfill(mask,'holes');
            obj = ml_findmainobj2d_bw(mask);
            cellbody{j}=obj;
        end

        cellcodes = {};
        for k = 1:1:length(cellbody)
                disp(['Calculating cell code on stack number: ' num2str(k) ]);
                %calculate projection here
                %let nucbody become a z-projection and calculate distance according to the projection
                mask = sum( segdna, 3 );
                mask(find( mask ~= 0)) = 255;
                mask = logical( mask );
                obj = ml_findmainobj2d_bw(mask);
                crossectionArea = nnz(mask);
                nucbody = obj;

                if ~isempty(nucbody)
                    cellcodes{k} = ml_parsecell({},cellbody{k},nucbody,...
                        1,size(segdna(:,:,k)),...
                        {'da','nucarea','nuccenter','nucmangle','nuchitpts',...
                        'nuccontour','nucellhitpts','nucdist','nucelldist','nucecc',...
                        'cellarea','cellcenter','cellmangle','cellcontour',...
                        'cellhitpts','celldist','cellecc'},0);
                end
         end
            
         %icaoberg april 5, 2012
         save(temporaryResultsFile, 'cellcodes', 'equatorZ', 'cellnucheightratio', 'nucbottomslice');
        catch
         disp( ['Ignoring image ' num2str(i)] );
        end
    end
end
end
