function[def_X, def_Y, Vx, Vy, distar] = imgReg_greedy_LDDMM(S, T, options)
% The input of this function is the source image 'S' and the target image 'T'.
% The input 'option' contains the parameters of LDDMM, and it is optional.
% If there is 'option' input, it will use the inputed one, if there is no
% 'option' input, the code will use the default parameters.
% The 'L = alpha*<> + gamma*Id' operator contains 'alpha' which is the
% coefficient of Lapacian operator and 'gamma' which is the coefficient of
% the identity matrix.
% The output of this funtion is the deformation field as well as velocity
% field, with def(x, t+1) = def(x+tau*V, t). Therefore, the size of the
% deformation field is [nx, ny, nt+1] and size of velocity field is [nx, ny, nt]

if ~exist('options','var') % Default parameters of LDDMM
    alpha = 0.8;          % coefficient of Lapacian operator
    gamma = 0.04;       % Coefficient of the Identity
    tau = 0.04;           % Step length of LDDMM;
    max_it = 75;        % Max iteration
else    % User inputed parameters
    alpha = options.alpha;
    gamma = options.gamma;
    tau = options.tau;
    max_it = options.max_it;    
end

  % initialization parameters
  [M, N] = size(S);
  [X_0,Y_0] = meshgrid(1:N, 1:M);
  dist = 0;
  src = S; % src is the updated source image in every iterations (Which is shown by cartoon)
  
  kkx = (2*pi/N)*[0:(N/2-1) (-N/2):(-1)];
  kky = (2*pi/M)*[0:(M/2-1) (-M/2):(-1)];
  [KX,KY]  = meshgrid(kkx,kky);                               
  delsq = 2*alpha*( (1-cos(KX))/(1/1^2) + (1-cos(KY))/(1/1^2))+ gamma;
  delsq = -delsq.^2;
  delsq(1,1) = 1;
  X_total1 = X_0; Y_total1 = Y_0; 
  distar = [];
  def_X(:,:,1) = X_0;   % def(:,0) = X0;
  def_Y(:,:,1) = Y_0;
  
for j = 1: max_it       
  %compute force F = -(T - src).*grad(src)
  [dIx1,dIy1] = gradient(src);
  im_diff = T - src; % Computing the difference in images
  Fx1 = -im_diff.*dIx1;
  Fy1 = -im_diff.*dIy1;		

  % Computing the velocity by solving (LtL)v = F for v
  fhat_x = fft2(Fx1);
  Vx1 = real(ifft2(fhat_x./delsq));
  Vx1 = Vx1 - Vx1(1,1);  % Specify arbitrary constant by forcing corner u = 0.
  fhat_y = fft2(Fy1);
  Vy1 = real(ifft2(fhat_y./delsq));
  Vy1 = Vy1 - Vy1(1,1);  % Specify arbitrary constant by forcing corner u = 0.
        
  Vx1(:,1) = 0;Vx1(:,N) = 0;Vx1(1,:) = 0;Vx1(M,:) = 0;
  Vy1(:,1) = 0;Vy1(:,N) = 0;Vy1(1,:) = 0;Vy1(M,:) = 0;
  
  % Computing the distance by (1/N*M)*sqrt(dx2+ dy2);
  dx2 = mean2((-del2(Vx1)*4*alpha + gamma*Vx1).^2);
  dy2 = mean2((-del2(Vy1)*4*alpha + gamma*Vy1).^2);
  Vx(:,:,j) = Vx1; Vy(:,:,j) = Vy1; % Record the velocity field.

  dist = dist + tau*sqrt(dx2 + dy2);
  distar = [distar, dist];  % Record the distance
  
  % Update the deformation field by Eularian reference
  X_c = X_0 + tau*Vx1;  
  Y_c = Y_0 + tau*Vy1;
  X_total1 = interp2(X_total1,X_c,Y_c,'*linear',0);
  Y_total1 = interp2(Y_total1,X_c,Y_c,'*linear',0);

  def_X(:,:,j+1) = X_total1;    % % Record the deformation field. 
  def_Y(:,:,j+1) = Y_total1;
  
  % Show the cartoon of registration process.
  src = interp2(S, X_total1, Y_total1);
  subplot(1,2,1), imshow(src, []);drawnow;
  curIt = num2str(j);
  title(curIt);
  subplot(1,2,2), imshow(T, []);drawnow;
  
end  

