function[distar1, distar2, wholeDis, img] = Greedy3D_lambda(S, T, lambda, options)
% The input of this function is the source image 'S' and the target image 'T'.
% The input 'option' contains the parameters of LDDMM, and it is optional.
% If there is 'option' input, it will use the inputed one, if there is no
% 'option' input, the code will use the default parameters.
% The 'L = alpha*<> + gamma*Id' operator contains 'alpha' which is the
% coefficient of Lapacian operator and 'gamma' which is the coefficient of
% the identity matrix.
% The output of this funtion is the deformation field as well as velocity
% field, with def(x, t+1) = def(x+tau*V, t). Therefore, the size of the
% deformation field is [nx, ny, nt+1] and size of velocity field is [nx, ny, nt]

% Author: Wei Wang
%
% Copyright (C) 2012 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

if ~exist('options','var') % Default parameters of LDDMM
    alpha = 1;          % coefficient of Lapacian operator
    beta = 0;           % coefficient of Gradient operator
    gamma = 0.04;       % Coefficient of the Identity
    tau = .01;           % Step length of LDDMM;
    max_it = 149;        % Max iteration
else    % User inputed parameters
    alpha = options.alpha;
    beta = options.beta;
    gamma = options.gamma;
    tau = options.tau;
    max_it = options.max_it;    
end

if ~isempty(find(lambda<0))
    disp('Lambda should be a positive number');
    lambda = abs(lambda);
end

  src = S;
  trg = T;
  src2 = src;
  trg2 = trg;
  % initialization parameters
  [M, N, P] = size(S);
  [X_0,Y_0,Z_0] = meshgrid(1:N, 1:M, 1:P);
  dist1 = 0; dist2 = 0;
  h = [0, -alpha-(beta/2), 0; -alpha-(beta/2), 4*alpha+gamma, -alpha+(beta/2);...
       0, -alpha+(beta/2), 0];
  H = fft2(h, M, N);
  H2 = -abs(H.^2);
  H2(1,1) = 1;

  X_total1 = X_0; Y_total1 = Y_0; 
  X_total2 = X_0; Y_total2 = Y_0;
  distar1 = [dist1];
  distar2 = [dist2];
  defs_X(:,:,1) = X_0;   % def(:,0) = X0;
  defs_Y(:,:,1) = Y_0;
  deft_X(:,:,1) = X_0;
  deft_Y(:,:,1) = Y_0;
  
for j = 1: max_it       
  %compute force F = -(T - src).*grad(src)
  [dIx1,dIy1] = gradient(src2);
  [dIx2,dIy2] = gradient(trg2);
  im_diff = trg2 - src2;% Computing the difference in images
  Fx1 = -im_diff.*dIx1;
  Fy1 = -im_diff.*dIy1;		
  Fx2 = im_diff.*dIx2;
  Fy2 = im_diff.*dIy2;	
  
  % Contribution of the source image
  % Computing the velocity by solving (LtL)v = F for v
  fhat_x = fft2(Fx1);
  Vx1 = real(ifft2(fhat_x./H2));
  Vx1 = Vx1 - Vx1(1,1);  % Specify arbitrary constant by forcing corner u = 0.
  fhat_y = fft2(Fy1);
  Vy1 = real(ifft2(fhat_y./H2));
  Vy1 = Vy1 - Vy1(1,1);  % Specify arbitrary constant by forcing corner u = 0.
        
  Vx1(:,1) = 0;Vx1(:,N) = 0;Vx1(1,:) = 0;Vx1(M,:) = 0;
  Vy1(:,1) = 0;Vy1(:,N) = 0;Vy1(1,:) = 0;Vy1(M,:) = 0;
  
  % Computing the distance by (1/N*M)*sqrt(dx2+ dy2);
%   dx2 = mean2((-del2(Vx1)*4*alpha + gamma*Vx1).^2);
%   dy2 = mean2((-del2(Vy1)*4*alpha + gamma*Vy1).^2);
  tx = imfilter(Vx1, h, 'replicate');
  ty = imfilter(Vy1, h, 'replicate');
  dx2 = mean2(tx(2:end-1,2:end-1).^2);
  dy2 = mean2(ty(2:end-1,2:end-1).^2);
  Vx(:,:,j) = Vx1; Vy(:,:,j) = Vy1; % Record the velocity field.

  dist1 = dist1 + tau*sqrt(dx2 + dy2);
  distar1 = [distar1, dist1];  % Record the distance
  
  % Update the deformation field by Eularian reference
  X_c = X_0 + tau*Vx1;  
  Y_c = Y_0 + tau*Vy1;
  X_total1 = interp2(X_total1,X_c,Y_c,'*linear',0);
  Y_total1 = interp2(Y_total1,X_c,Y_c,'*linear',0);

  defs_X(:,:,j+1) = X_total1;    % % Record the deformation field. 
  defs_Y(:,:,j+1) = Y_total1;
  
  % contribution of target image
  
  fhat_x = fft2(Fx2);
  Vx2 = real(ifft2(fhat_x./H2));
  Vx2 = Vx2 - Vx2(1,1);  % Specify arbitrary constant by forcing corner u = 0.
  
  fhat_y = fft2(Fy2);
  Vy2 = real(ifft2(fhat_y./H2));
  Vy2 = Vy2 - Vy2(1,1);  % Specify arbitrary constant by forcing corner u = 0.
        
  Vx2(:,1) = 0;Vx2(:,N) = 0;Vx2(1,:) = 0;Vx2(M,:) = 0;
  Vy2(:,1) = 0;Vy2(:,N) = 0;Vy2(1,:) = 0;Vy2(M,:) = 0;
  
%   dx22 = mean2((-del2(Vx2)*4*alpha + gamma*Vx2).^2);
%   dy22 = mean2((-del2(Vy2)*4*alpha + gamma*Vy2).^2);
  dx22 = mean2(imfilter(Vx2, h, 'replicate').^2);
  dy22 = mean2(imfilter(Vy2, h, 'replicate').^2);

  dist2 = dist2 + tau*sqrt(dx22 + dy22);
  distar2 = [distar2,dist2];
 
  X_c = X_0 + tau*Vx2;
  Y_c = Y_0 + tau*Vy2;
  
  X_total2 = interp2(X_total2,X_c,Y_c,'*linear',0);
  Y_total2 = interp2(Y_total2,X_c,Y_c,'*linear',0);
  
  deft_X(:,:,j+1) = X_total2;    % % Record the deformation field. 
  deft_Y(:,:,j+1) = Y_total2;  
  
  % Show the cartoon of registration process.
  src2 = interp2(S, X_total1, Y_total1);
  trg2 = interp2(T,X_total2,Y_total2);
  figure(999);
  subplot(1,2,1), imshow(src2, []);drawnow;
  subplot(1,2,2), imshow(trg2, []);drawnow;
  curIt = num2str(j);
  title(curIt);
  
end  

wholeDis = distar1(end) + distar2(end);
if length(lambda)==1
    if lambda >= distar1(end)/distar2(end)
        [value,ind] = min(abs(distar1 - (1/(1+lambda))*wholeDis));
        img = interp2(S, defs_X(:,:,ind), defs_Y(:,:,ind));
    else
        [value,ind] = min(abs(distar1 - (lambda/(1+lambda))*wholeDis));
        img = interp2(T, deft_X(:,:,ind), deft_Y(:,:,ind));    
    end
    %figure(998), imshow(img,[]);
else
    img(:,:,1) = S;
    for kk = 1:length(lambda)
        if lambda(kk) >= distar1/distar2
            [value,ind] = min(abs(distar1 - (1/(1+lambda(kk)))*wholeDis));
            img(:,:,kk+1) = interp2(S, defs_X(:,:,ind), defs_Y(:,:,ind));
        else
            [value,ind] = min(abs(distar1 - (lambda(kk)/(1+lambda(kk)))*wholeDis));
            img(:,:,kk+1) = interp2(T, deft_X(:,:,ind), deft_Y(:,:,ind));    
        end
    end
    img(:,:,end+1) = T;
end


