function [shape_space] = compute_shape_space_model_given_images(options)
% Build a shape space model from images. Takes as input only an options structure and returns a model structure. This function can be called with the same options by multiple jobs to do certain parts in parallel.
% Assumptions:
% Images are the same size and are square in XY.
% 
% Options you must specify for good results:
% image_function
% convergence_absolute_error, the sum of the absolute differences between any two images at which interpolation convergence will be assumed. Specifically, during an interpolation between a source and a target image, the images will be iteratively deformed towards each other. After each iteration, this convergence test will be done.
% voxel_size
% shape_aspect_ratio
% save_location
% 
% Options you should specify for completeness:
% 
% 
% Other options can be seen in the default_options structure at the beginning of this function.
% 
% Dependencies from File Exchange:
% % inhull
% % sort_nat
% % export_fig
% % CompressionLib
% % ode2stab
% % convnfft
% % DataHash
% % Dependencies from the Murphy and Rohde labs:
% % ./renuclearmorphometrycode
% % ./Taraz
% % /home/icaoberg.projects/slic
% 
% History:
% 2013-02-13 tebuck: Copied from test_walk_ternary025.m.
% 
% To do:
% Use MDS approximations.
% Allow using meshes instead of images.


  % We require certain options:
  required_options = {'image_function', 'convergence_absolute_error', 'voxel_size', 'shape_aspect_ratio', 'save_location', 'number_images'};
  for index = 1:length(required_options)
    required_option = required_options{index};
    if ~isfield(options, required_option)
      error('Option %s is required', required_option)
    end
  end


  first_image = options.image_function(1);

  [M, N, P] = size(first_image)
  [X_0, Y_0, Z_0] = meshgrid(1:N, 1:M, 1:P);
  identity_map = {X_0, Y_0, Z_0}; 
  clear X_0 Y_0 Z_0
  
  window_size = [M, N, P];




  
  % Default options go in a structure passed to process_options_structure:
  default_options = struct();
  
  
  % Parameters for windowed LDDMM (these should probably be in a substructure so field names do not overlap (not that they should as of 2013-02-14)):
  default_options.filter_radius = 32;
  % default_options.window_radius = default_options.filter_radius * 2;
  default_options.window_radius = window_size(1);
  % Flat cells:
  default_options.kernel_z_radius = default_options.filter_radius * options.shape_aspect_ratio(3);
  % Flat cells:
  default_options.maximum_deformation_per_step = [1, 1, .5];

  default_options.maximum_iterations = 1000;
  
  default_options.just_compute_distance = true;

  % Allow option of early stopping during interpolations:
  default_options.use_known_distances = false;


  % Options specific to this function:
  default_options.maximum_shape_space_dimension = 7;
  default_options.explained_variance_threshold = .9;
  
  
  % Combine default and given options:
  options = process_options_structure(default_options, options); 
  
  
  base_filename = [options.save_location, filesep];

  
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % parallel: Register images to mean image for computing distance matrix approximation:
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  distances_filename = [base_filename, 'distances'];
  % comparison_filename_prefix = distances_filename;
  comparison_filename_prefix = base_filename;
  [distances, success] = compute_distance_matrix(...
    options.image_function, ...
    options.number_images, comparison_filename_prefix, options, @(s, t)0 ...
    );
  if ~success
    error('Cannot continue, some unfinished distances are being computed elsewhere.')
  end


  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  % serial: Compute shape space:
  %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

  % Limit based on rule of thumb at <http://www.qhull.org/html/qh-code.htm>:
  maximum_shape_space_dimension = min(options.maximum_shape_space_dimension, 7);
  explained_variance_threshold = options.explained_variance_threshold;

  shape_space_filename = [base_filename, 'shape_space'];
  [result, success] = compute_shape_space(...
    distances, 0, ...
    [explained_variance_threshold, maximum_shape_space_dimension], shape_space_filename...
    );
  if ~success
    error('Cannot continue, shape space being computed elsewhere.')
  end
  y2 = result.y2;
  convex_hull = result.convex_hull;
  tes = result.tes;


  shape_space = struct();
  shape_space.positions = y2;
  shape_space.convex_hull = convex_hull;
  shape_space.tessellation = tes;
  shape_space.distances = distances;

end
