% demo3D11
%
% Trains a generative model of the cell framework using the four patterns in the HeLa
% dataset

% Created: Ivan E. Cao-Berg
%
% Copyright (C) 2012 Murphy Lab
% Lane Center for Computational Biology
% School of Computer Science
% Carnegie Mellon University
%
% This program is free software; you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published
% by the Free Software Foundation; either version 2 of the License,
% or (at your option) any later version.
%
% This program is distributed in the hope that it will be useful, but
% WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
% General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program; if not, write to the Free Software
% Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
% 02110-1301, USA.
%
% For additional information visit http://murphylab.web.cmu.edu or
% send email to murphy@cmu.edu

curr_path = which('demo3D11.m');
curr_path = curr_path(1:end-10);
cd(curr_path);

%list of available patterns
patterns = { 'Nuc', 'Mit', 'LAM', 'TfR' };

%icaoberg 11/16/2012
url = 'http://murphylab.web.cmu.edu/data/Hela/3D';

imageDirectory = [ pwd filesep 'images' ];
if ~exist( imageDirectory, 'dir' )
    mkdir( imageDirectory );
end

if exist( imageDirectory, 'dir' ) && length( dir(imageDirectory) ) == 196
    disp('Image directory found and built');
else
    for index=1:1:length(patterns)
        pattern = patterns{index};
        disp( pattern )
        tarball = [ '3DHela_' pattern '.tgz' ];
        if ~exist( tarball, 'file' )
            disp([ 'Downloading Dataset: ' tarball ]);
            urlwrite([url filesep tarball], tarball);
            
            disp( 'Extracting files' );
            system(['tar -xvf ' tarball] );
        else
            disp( 'Pattern found on disk. Extracting files from existing tarball' );
            if ~exist( pattern, 'file' )
                disp( 'Extracting files' );
                untar( tarball );
            else
                disp( 'Extracted folder found. Skipping extraction' );
            end
        end
    end
    
    for i=1:1:length(patterns)
        pattern = patterns{i};
        folders = dir([pattern filesep 'cell*']);
        for j=3:1:length(folders)
            folder = folders(j).name;
            disp([pattern ':' folder]);
            system(['cp -r ' pattern filesep folder ' images' filesep folder num2str(i)]);
        end
    end
end

dimensionality = '3D';
pattern = 'framework';

dna = imageDirectory;
cell = imageDirectory;
protein = imageDirectory;

% generic model options
% ---------------------
% model.name                (optional) Holds the name of the model. Default is empty.
param.model.name = 'all';

% model.id                  (optional) Holds the id of the model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.model.id = result(1:end-1);
catch err
    param.model.id = num2str(now);
end

% model.filename            Holds the output filename.
param.model.filename = [ lower(pattern) '.xml' ];

% nuclear shape model options
% ---------------------------
% nucleus.type              Holds the nuclear model type. Default is "medial axis".
param.nucleus.type = 'medial axis';

% nucleus.name              (optional) Holds the name of the nuclear model. Default is empty.
param.nucleus.name = 'all';

% nucleus.id                (optional) Holds the id of the nuclear model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.nucleus.id = result(1:end-1);
catch err
    param.nucleus.id = num2str(now);
end

% cell shape model options
% ------------------------
% cell.type                 Holds the cell model type. Default is "ratio".
param.cell.type = 'ratio';

% cell.name                 (optional) Holds the name of the cell model. Default is empty.
param.cell.model = 'framework';

% cell.id                   (optional) Holds the id the cell model. Default is empty.
try
    [status,result] = system( 'uuidgen' );
    param.cell.id = result(1:end-1);
catch err
    param.cell.id = num2str(now);
end

% other options
% -------------
% verbose                   (optional) Displays messages to screen. The default is true.
param.verbose = true;

% debug                     (optional) Reports errors and warnings. Default is false.
param.debug = true;

% train.flag                (optional) Selects what model is going to be trained ('nuclear',
%                           'framework', or 'all'). Default is 'all'
param.train.flag = 'framework';

%documentation
param.documentation.author = 'Murphy Lab';
param.documentation.email = 'murphy@cmu.edu';
param.documentation.description = 'This is the framework trained on all images from the HeLa dataset.';

answer = img2slml( dimensionality, dna, cell, protein, param );
